/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: cifXFunctions.cpp 6424 2014-07-01 12:15:23Z stephans $
   Last Modification:
    $Author: stephans $
    $Date: 2014-07-01 14:15:23 +0200 (Di, 01 Jul 2014) $
    $Revision: 6424 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
    Implementation of the cifX Driver API
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
      6        14.12.11    RM       Added stub functions for xSysdeviceBootstart() and
                                    xSysdeviceExtendedMemory()
      5        25.11.11    SS       - xDriverRestartDevice failed due to wrong parameter validation
      4        18.04.11    SS       - Review validation of pointers and handles passed to cifX API functions
                                    - Add stubbed cifX API functions xChannelSyncState,
                                      xChannelRegisterNotification, xChannelUnregisterNotification
      3        31.03.11    SS       Remove of physical layer now done by Destructor
                                    of device handler class
      2        15.12.10    RM       nxDrvFindDevice()
                                    - now calls nxDrvBrowseDevices() on NXDRV_FIND_FIRST because
                                    nxDRvInit() so not calling browse devices anymore to speedup
                                    the DLL load.
                                    - fixed incrementation of pulSearchIdx

      1        03.03.09    MT       initial version
 
**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file cifXFunctions.cpp
/// Implementation of cifX Driver API
/////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#include <stdio.h>
#include <atlstr.h>
#include "io.h"
#include "direct.h"
#include "Rpc.h"

#include "stdint.h"

#include "rcX_User.h"
#include "CifxUser.h"
#include "CIFxErrors.h"
#include "netXAPI.h"

#include "DeviceHandler.h"
#include "TransportLayer.h"
#include "PhysicalInterface.h"
#include "ConnectorAPI.h"

#include "HelperFunctions.h"

/* include class to determine a full file version information */
#include "verinfo.h"

/* Gloabl DLL definitions */
extern HMODULE        g_hModule;                                                //!< External modul 

// Handling of connector dll
typedef               vector<NETX_CONNECTOR_DLL_INFO_T*> NX_CONNECTOR_DLL_LIST; //!< Definition of connector vector       
NX_CONNECTOR_DLL_LIST g_cvConnectorDllList;                                     //!< Vector of available connector DLLs   

// Connection definitions 
CDeviceHandler*       g_pcDeviceHandler         = NULL;                         //!< Global device handler object 
LONG                  g_lDriverOpenCnt          = 0;                            //!< Global open counter of the driver object 
CRITICAL_SECTION      g_tDriverOpenLock;

// Handling of nxAPI calls and devices 
typedef map< HANDLE, NXDRV_DEVICE_INFORMATION>  NXAPI_DEVICE_INFO_LIST;          //!< Definition of the device info                  
NXAPI_DEVICE_INFO_LIST  g_cmNxDevInfoList;                                        //!< Map of the available devices 

// Diagnostics driver information
static DWORD s_ulDriverType = 0;

#define CHECK_POINTER(param) if ((void*)NULL == param) return CIFX_INVALID_POINTER;
#define CHECK_DRIVERHANDLE(handle) if ((NULL == handle) || (g_pcDeviceHandler != handle)) return CIFX_INVALID_HANDLE;
#define CHECK_SYSDEVICEHANDLE(handle) if (CheckSysdeviceHandle(handle) != CIFX_NO_ERROR) return CheckSysdeviceHandle(handle);
#define CHECK_CHANNELHANDLE(handle) if (CheckChannelHandle(handle)!= CIFX_NO_ERROR) return CheckChannelHandle(handle);

/*****************************************************************************/
/*! Checks if the given channel handle is valid
*   \param hDriver      Channel handle
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t CheckChannelHandle(CIFXHANDLE hChannel)
{
  CDevice*      pcDevice = NULL;
  uint32_t ulDev    = 0;
  int32_t          lRet     = CIFX_INVALID_HANDLE;

  if (!g_pcDeviceHandler)
    return CIFX_DRV_NOT_OPENED;

  while (NULL != (pcDevice = g_pcDeviceHandler->GetDevice(ulDev++, false)))
  {
    CChannel*     pcChannel = NULL;
    uint32_t ulChannel = 0;

    while (NULL != (pcChannel = pcDevice->GetChannelObject(ulChannel++)))
    {
      if (hChannel == pcChannel)
      {
        lRet = CIFX_NO_ERROR;
        break;
      }
    }
    if (CIFX_NO_ERROR == lRet)
      break;
  }
  
  return lRet;
}

/*****************************************************************************/
/*! Checks if the given channel handle is valid
*   \param hDriver      Channel handle
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t CheckSysdeviceHandle(CIFXHANDLE hChannel)
{
  CDevice*      pcDevice = NULL;
  uint32_t ulDev    = 0;
  int32_t          lRet     = CIFX_INVALID_HANDLE;

  if (!g_pcDeviceHandler)
    return CIFX_DRV_NOT_OPENED;

  while (NULL != (pcDevice = g_pcDeviceHandler->GetDevice(ulDev++, false)))
  {
    CChannel* pcChannel = pcDevice->GetChannelObject(RCX_SYSTEM_CHANNEL);

    if ((NULL != pcChannel) && (hChannel == pcChannel))
    {
      lRet = CIFX_NO_ERROR;
      break;
    }
  }
  
  return lRet;
}

/***************************************************************************
* 
*   
*   Connector Support Functions....................  
*
*
******************************************************************************/
/////////////////////////////////////////////////////////////////////////////
/// Convert UUID from UUID struct to string 
///   \param    tUUID           UUID structur
///   \param    szConnectorUUID Reference of UUID string
///   \return   NXAPI_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
void HandelUUIDtoString( UUID tUUID, char* szConnectorUUID)
{
  CString szUUIDString;

  szUUIDString.Format(_T("%X-%X-%X-%X%X-%X%X%X%X%X%X"), tUUID.Data1,tUUID.Data2,
                                                        tUUID.Data3,tUUID.Data4[0],
                                                        tUUID.Data4[1],tUUID.Data4[2],
                                                        tUUID.Data4[3],tUUID.Data4[4],
                                                        tUUID.Data4[5],tUUID.Data4[6],
                                                        tUUID.Data4[7]);
  
  strncpy(szConnectorUUID, szUUIDString, szUUIDString.GetLength());
}

/////////////////////////////////////////////////////////////////////////////////////////
///  UnloadConnectorDLL
///   \param    ptConnectorInfo Information structure of the connector dll
///   \return   void
/////////////////////////////////////////////////////////////////////////////////////////
void UnloadConnectorDLL( NETX_CONNECTOR_DLL_INFO_T* ptConnectorInfo)
{
  // Clean-up the driver
  if ( NULL != ptConnectorInfo)
  {
    FreeLibrary(ptConnectorInfo->hDll);
    ptConnectorInfo->hDll = NULL;
  }
}

/////////////////////////////////////////////////////////////////////////////////////////
/// LoadConnectorDLL
///   \param ptConnectorInfo Connector Information structure
///   \return true on success
/////////////////////////////////////////////////////////////////////////////////////////
bool LoadConnectorDLL( NETX_CONNECTOR_DLL_INFO_T* ptConnectorInfo)
{
  bool fRet = false;

  //---------------------------------------
  // Read the cifX API
  //---------------------------------------
  if ( NULL == (ptConnectorInfo->hDll = LoadLibrary( ptConnectorInfo->szFullFileName)))
  {
    // Unable to load connector DLL
    ptConnectorInfo->ulError = GetLastError();
  } else
  {
    // Read the netX API driver functions from the driver DLL
    NETX_CONNECTOR_FUNCTION_TABLE* pfnFnc = &ptConnectorInfo->tFncTable;
    
    if ( ((pfnFnc->pfnNetXConOpen               = ((PFN_NETXCON_OPEN)                 GetProcAddress( ptConnectorInfo->hDll, "netXConOpen")))                      == NULL) ||
         ((pfnFnc->pfnNetXConClose              = ((PFN_NETXCON_CLOSE)                GetProcAddress( ptConnectorInfo->hDll, "netXConClose")))                     == NULL) ||
         ((pfnFnc->pfnNetXConCreateInterface    = ((PFN_NETXCON_CREATEINTERFACE)      GetProcAddress( ptConnectorInfo->hDll, "netXConCreateInterface")))           == NULL) ||
         ((pfnFnc->pfnNetXConGetInformation     = ((PFN_NETXCON_GETINFORMATION)       GetProcAddress( ptConnectorInfo->hDll, "netXConGetInformation")))            == NULL) ||
         ((pfnFnc->pfnNetXConGetConfig          = ((PFN_NETXCON_GETCONFIG)            GetProcAddress( ptConnectorInfo->hDll, "netXConGetConfig")))                 == NULL) ||
         ((pfnFnc->pfnNetXConSetConfig          = ((PFN_NETXCON_SETCONFIG)            GetProcAddress( ptConnectorInfo->hDll, "netXConSetConfig")))                 == NULL) ||
         ((pfnFnc->pfnNetXConIntfStart          = ((PFN_NETXCON_INTF_START)           GetProcAddress( ptConnectorInfo->hDll, "netXConStartInterface")))            == NULL) ||
         ((pfnFnc->pfnNetXConIntfStop           = ((PFN_NETXCON_INTF_STOP)            GetProcAddress( ptConnectorInfo->hDll, "netXConStopInterface")))             == NULL) ||
         ((pfnFnc->pfnNetXConIntfSend           = ((PFN_NETXCON_INTF_SEND)            GetProcAddress( ptConnectorInfo->hDll, "netXConSendInterface")))             == NULL) ||
         ((pfnFnc->pfnNetXConIntfGetInformation = ((PFN_NETXCON_INTF_GETINFORMATION)  GetProcAddress( ptConnectorInfo->hDll, "netXConGetInformationInterface")))   == NULL) ||
         ((pfnFnc->pfnNetXConCreateDialog       = ((PFN_NETXCON_CREATEDIALOG)         GetProcAddress( ptConnectorInfo->hDll, "netXConCreateDialog")))              == NULL) ||
         ((pfnFnc->pfnNetXConEndDialog          = ((PFN_NETXCON_ENDDIALOG)            GetProcAddress( ptConnectorInfo->hDll, "netXConEndDialog")))                 == NULL)  )

    {
      // Error reading the connector functions from the DLL
      ptConnectorInfo->ulError  = (uint32_t)CIFX_CONNECTOR_FUNCTIONS_READ_ERROR;
    }else
    {
      char  abIdentifier[NXCON_MAX_LENGTH_CONNECTOR_IDENTIFIER] = {0};
      UUID                  tUUID = {0};
      
      // Check supported identifier of connector dll
      if ( (NXCON_NO_ERROR != ptConnectorInfo->tFncTable.pfnNetXConGetInformation( eIDENTIFIER, sizeof(abIdentifier), abIdentifier)) ||
           (NXCON_NO_ERROR != ptConnectorInfo->tFncTable.pfnNetXConGetInformation( eUUID, sizeof(tUUID), &tUUID)) )
      {
        ptConnectorInfo->ulError = (uint32_t)CIFX_CONNECTOR_IDENTIFIER_EMPTY;
      } else
      {
        // Check length of identifier
        if (strlen( abIdentifier) == 0)                  
        {
          ptConnectorInfo->ulError = (uint32_t)CIFX_CONNECTOR_IDENTIFIER_EMPTY;
        } else
        { 
          for (uint32_t ulCount=0; ulCount < g_cvConnectorDllList.size(); ulCount++)
          {
            // Check for duplicate connector identifers
            if( 0 == strcmp( g_cvConnectorDllList[ulCount]->szConIdentifier, abIdentifier))
            {
              ptConnectorInfo->ulError  = (uint32_t)CIFX_CONNECTOR_DUPLICATE_IDENTIFIER;
              break;
            }
          }

          if( CIFX_NO_ERROR == ptConnectorInfo->ulError)
          {
            // Save connector identifier and UUID
            strncpy( ptConnectorInfo->szConIdentifier,  abIdentifier, strlen(abIdentifier));
            HandelUUIDtoString(tUUID, ptConnectorInfo->szConnectorUUID);
            // Query and store description and connector type
            ptConnectorInfo->tFncTable.pfnNetXConGetInformation( eTYPE, sizeof(ptConnectorInfo->tType), &ptConnectorInfo->tType);
            if (NXCON_NO_ERROR != ptConnectorInfo->tFncTable.pfnNetXConGetInformation( eDESCRIPTION, sizeof(ptConnectorInfo->szDescription), ptConnectorInfo->szDescription))
            {
              strcpy(ptConnectorInfo->szDescription, ptConnectorInfo->szFileName);
            }
          }
        }
      }
    }

    if( CIFX_NO_ERROR != ptConnectorInfo->ulError)
    {
      // Something is wrong, don't load the DLL
      UnloadConnectorDLL( ptConnectorInfo);
    } else
    {
      // Successfully opend and read the functions
      fRet = true;
    }
  }

  return fRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Get plugin directory containing the connector libraries
/// \param ulSize       Size of buffer for plugin path string
/// \param szPluginPath String buffer to return plugin path
/// \return NXAPI_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t GetPluginPath( uint32_t ulSize, char* szPluginPath )
{
  // Read actual path by getting the directory of this dll
  int32_t lRet                    = NXAPI_NO_ERROR;
  char szModulePath[_MAX_PATH] = {0};  

  if ( 0 == GetModuleFileName( g_hModule, szModulePath, sizeof(szModulePath)))
  {
    lRet = NXAPI_NO_WORKING_DIRECTORY;

  } else
  {
    // We have a valid full file name, extract the path from it
    char szDrive[_MAX_DRIVE]   = {0};
    char szDirectory[_MAX_DIR] = {0};
    
    _splitpath(  szModulePath, szDrive, szDirectory, NULL, NULL);
                                   
    memset( szPluginPath, 0, ulSize);
    strcat( szPluginPath, szDrive);
    strcat( szPluginPath, szDirectory);   
	  strcat( szPluginPath, NXCON_DLL_PLUGIN_DIR);
    strcat( szPluginPath, _T("\\"));
  }
  
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Search the available driver DLLs
///   \param  szPluginPath    Plugin path to search for connector libraries
///   \param  pvConnectorList List to return found connector libraries
///   \return NXAPI_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t SearchConnectorDlls( char *szPluginPath, NX_CONNECTOR_DLL_LIST* pvConnectorList)
{
  int32_t lRet = NXAPI_NO_ERROR;
  
  //--------------------------------------
  // Search interface DLLs and load driver
  //--------------------------------------
  if ( 0 == strlen(szPluginPath))
    return NXAPI_NO_WORKING_DIRECTORY;
  
  char szSearchFile[1024] = {0};

  strcpy(szSearchFile, szPluginPath);
	strcat(szSearchFile, NXCON_DLL_SEARCH_STRING);
  
  // Create search string with path information
	intptr_t  hFile;
	struct    _tfinddata_t tFile;
	
  if( (hFile = _tfindfirst( szSearchFile, &tFile)) != -1L )
	{
    // DLL found
    do 
    {
      if ( !(tFile.attrib & _A_SUBDIR) ) 
      {
        //---------------------------------------
        // Connector DLL found
        //---------------------------------------
        // Check if this is a connector DLL and load interface
        NETX_CONNECTOR_DLL_INFO_T* ptConnectorInfo = (NETX_CONNECTOR_DLL_INFO_T*)new char[sizeof(NETX_CONNECTOR_DLL_INFO_T)];
        memset( ptConnectorInfo, 0, sizeof(*ptConnectorInfo));

        // Store information into info structure
        strcat(ptConnectorInfo->szFileName, tFile.name);
        strcat(ptConnectorInfo->szFullFileName, szPluginPath);
        strcat(ptConnectorInfo->szFullFileName, tFile.name);

        // Try to load the connector
        if  ( (true == LoadConnectorDLL( ptConnectorInfo)) &&
              (   0 == ptConnectorInfo->ulError)              )
        {
          /* We have a new connector, save to connector list */
          pvConnectorList->push_back( ptConnectorInfo);
        } else
        {
          delete [] ptConnectorInfo;
          ptConnectorInfo = NULL;
        }
      }
    } while( _tfindnext( hFile, &tFile) == 0 );

    _findclose( hFile);
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// CheckHandle
///   \param    pcChannel Channel handle
///   \return   true if channel is available
/////////////////////////////////////////////////////////////////////////////
static bool CheckHandle(CChannel* pcChannel)
{
  bool      fRet     = false;
  CDevice*  pcDevice = pcChannel->m_pcDevice;

  if(pcChannel->IsAvailable())
  { 
    fRet = true;

  } else if(pcDevice->IsAvailable())
  {
    /* Device is there, but channel is not known, so the reconnect
       succeeded, but the channel vanished */
    fRet = false;

  } else if(pcDevice->m_pcEndpoint->m_pcDefaultDataLayer != NULL)
  {
    /* This endpoint has already reconnected */
    fRet = false;
  } else
  {
    /* Channel is not valid, so QueueReconnect and return error */
    g_pcDeviceHandler->ScheduleInterfaceMonitor(pcDevice->m_pcEndpoint->m_pcInterface->GetInterfaceName(), CDeviceHandler::eRECONNECT);

    fRet = false;
  }

  return fRet;
}

/***************************************************************************
* 
*   
*   DRIVER FUNCTIONS....................  
*
*
*
*
*
******************************************************************************/

/*! **************************************************************************
* Opens the Driver
*   \param phDriver Returned handle to the driver
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xDriverOpen(HANDLE* phDriver)
{
  int32_t lRet = CIFX_NO_ERROR;

  CHECK_POINTER(phDriver);

  EnterCriticalSection(&g_tDriverOpenLock);
  if(g_lDriverOpenCnt++ == 0)
  {
    // Driver is opened for the first time
    char          szPluginPath[MAX_PATH] = {0};

    // Query pluging directory containing the connector libraries,
    // search for connector libraries and load them
    if ( (NXAPI_NO_ERROR == (lRet= GetPluginPath(sizeof(szPluginPath), szPluginPath)))        &&
         (NXAPI_NO_ERROR == (lRet= SearchConnectorDlls( szPluginPath, &g_cvConnectorDllList)))  )
    {
      // First open 
      g_pcDeviceHandler = new CDeviceHandler();

      if(CIFX_NO_ERROR == (lRet = g_pcDeviceHandler->Init()))
      {
        *phDriver = (HANDLE)g_pcDeviceHandler;

        // Add connectors 
        for( NX_CONNECTOR_DLL_LIST::iterator iterCon = g_cvConnectorDllList.begin(); iterCon != g_cvConnectorDllList.end(); iterCon++)
        {
          CPhysicalLayer* pcPhysLayer = new CPhysicalLayer(*iterCon);

          if( pcPhysLayer) 
          { 
            // Query driver type from connector
            s_ulDriverType = (s_ulDriverType | pcPhysLayer->GetType()); 

            // Add physical layer to the device handler managment      
            lRet =  g_pcDeviceHandler->AddLayer(pcPhysLayer);
            if( NXCON_DRV_DISABLED == lRet)
            {
              // Skip disabled connectors */
              lRet = CIFX_NO_ERROR;
            }
          }
        }
      }
    }

  } else
  { 
    // Driver is already open, just return the handle
    *phDriver = (HANDLE)g_pcDeviceHandler; 
  }
  LeaveCriticalSection(&g_tDriverOpenLock);

  return lRet;            
}

/*! **************************************************************************
* Closes a previously opened driver
*   \param hDriver Driver handle
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xDriverClose(HANDLE hDriver)
{ 
  int32_t lRet = CIFX_NO_ERROR;

  CHECK_DRIVERHANDLE(hDriver);

  EnterCriticalSection(&g_tDriverOpenLock);
  if(g_lDriverOpenCnt == 0)
  {
    lRet = CIFX_DRV_NOT_OPENED;
  } else if (g_lDriverOpenCnt-- == 1)
  {
    // Last close 
    delete g_pcDeviceHandler;
    g_pcDeviceHandler = NULL;

    // Remove Device list 
    while ( 0 != g_cvConnectorDllList.size())
    {
      NETX_CONNECTOR_DLL_INFO_T* ptConnectorInfo = g_cvConnectorDllList.front();
      
      // Unload connector dll
      UnloadConnectorDLL( ptConnectorInfo);

      delete [] ptConnectorInfo;
      g_cvConnectorDllList.erase(g_cvConnectorDllList.begin());
    } 
  }
  LeaveCriticalSection(&g_tDriverOpenLock);

  return lRet;
}

/*! **************************************************************************
* Restart a device
*   \param hDriver      Driver handle, if NULL the driver will be temporarily opened by the API
*   \param szBoardName  Board name
*   \param pvData       
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xDriverRestartDevice(CIFXHANDLE  hDriver, char* szBoardName, void* pvData)
{
  UNREFERENCED_PARAMETER(pvData); // Keep compiler happy
  CDevice* pcDevice = NULL;
  int32_t  lRet     = CIFX_NO_ERROR;

  if (g_pcDeviceHandler == NULL)
  {
    lRet = xDriverOpen(&hDriver);
  } else
  {
    hDriver = NULL;
  }

  if (CIFX_NO_ERROR == lRet)
  {
    pcDevice = g_pcDeviceHandler->GetDevice(szBoardName);
    if (NULL == pcDevice)
    {
      lRet = CIFX_INVALID_BOARD;
    } else
    {
      lRet = pcDevice->m_pcEndpoint->m_pcDefaultDataLayer->RestartDevice(pcDevice->m_tBoardInfo.abBoardName);
    }

    if(hDriver)
      xDriverClose( hDriver);
  }

  return lRet;
}

/*! **************************************************************************
* Query information about the running device driver
*   \param hDriver      Driver handle, if NULL the driver will be temporarily opened by the API
*   \param ulSize       Buffer size
*   \param pvDriverInfo Returned Driver information
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xDriverGetInformation(  HANDLE        hDriver, 
                                      uint32_t ulSize, 
                                      void*         pvDriverInfo)
{
  if (g_lDriverOpenCnt == 0)
    return CIFX_DRV_NOT_OPENED;

  // Check incomming parameter
  CHECK_DRIVERHANDLE(hDriver);
  CHECK_POINTER(pvDriverInfo);

  int32_t lRet = CIFX_NO_ERROR;

  DRIVER_INFORMATION* ptDriverInfo = (DRIVER_INFORMATION*)pvDriverInfo;
  if(ulSize >= sizeof(ptDriverInfo))
  {
    CFileVersionInfo cVerInfo;
    
    if (cVerInfo.Open(g_hModule))
    {
      char szDriverName[NXDRV_NAME_LENGTH];
      char szVersion[NXDRV_NAME_LENGTH];     
      cVerInfo.QueryStringValue(VI_STR_FILEVERSION, szVersion, sizeof(szVersion)/sizeof(szVersion[0]));
      cVerInfo.QueryStringValue(VI_STR_PRODUCTNAME, szDriverName, sizeof(szDriverName)/sizeof(szDriverName[0]));
      cVerInfo.Close();

      // Build driver version string
      _snprintf(ptDriverInfo->abDriverVersion, sizeof(ptDriverInfo->abDriverVersion), "%s V%s", szDriverName, szVersion);
    }

    ptDriverInfo->ulBoardCnt = g_pcDeviceHandler->GetDeviceCount(); // Use interface count

  } else
  {
    lRet = CIFX_INVALID_BUFFERSIZE;
  }

  return lRet;
}

/*! **************************************************************************
* Enumerate over all available Boards
*   \param hDriver      Driver handle, if NULL the driver will be temporarily opened by the API
*   \param ulBoard      Should be counted from 0 upwards on enumeration
*   \param ulSize       Size of the user buffer
*   \param pvBoardInfo  Returned board information
*   \return CIFX_NO_ERROR on success, CIFX_NO_MORE_ENTRIES when ulBoard is unknown
******************************************************************************/
int32_t APIENTRY xDriverEnumBoards(  HANDLE        hDriver,
                                  uint32_t ulBoard, 
                                  uint32_t ulSize,
                                  void*         pvBoardInfo)
{
  if(g_lDriverOpenCnt == 0)
    return CIFX_DRV_NOT_OPENED;
  
  CHECK_DRIVERHANDLE(hDriver);
  CHECK_POINTER(pvBoardInfo);  

  int32_t                  lRet        = CIFX_NO_ERROR;
  BOARD_INFORMATION*    ptBoardInfo = (BOARD_INFORMATION*)pvBoardInfo;
  
  if(ulSize < (uint32_t)sizeof(*ptBoardInfo))
  {
    lRet = CIFX_INVALID_BUFFERSIZE;
  } else
  {
    lRet = g_pcDeviceHandler->EnumerateDevice( ulBoard, ptBoardInfo);
  }

  return lRet;
}

/*! **************************************************************************
* Enumerate over all available Channels on a board
*   \param hDriver        Driver handle, if NULL the driver will be temporarily opened by the API
*   \param ulBoard        The board number during board enumeration
*   \param ulChannel      Should be counted from 0 upwards on enumeration
*   \param ulSize         Size of the user buffer
*   \param pvChannelInfo  Returned channel information
*   \return CIFX_NO_ERROR on success, CIFX_NO_MORE_ENTRIES when ulChannel is unknown
******************************************************************************/
int32_t APIENTRY xDriverEnumChannels(  HANDLE        hDriver,
                                    uint32_t ulBoard, 
                                    uint32_t ulChannel, 
                                    uint32_t ulSize, 
                                    void*         pvChannelInfo)
{
  if(g_lDriverOpenCnt == 0)
    return CIFX_DRV_NOT_OPENED;
  
  CHECK_DRIVERHANDLE(hDriver);
  CHECK_POINTER(pvChannelInfo);  

  int32_t                  lRet          = CIFX_NO_ERROR;
  CHANNEL_INFORMATION*  ptChannelInfo = (CHANNEL_INFORMATION*)pvChannelInfo;

  if(ulSize < (uint32_t)sizeof(*ptChannelInfo))
  {
    return CIFX_INVALID_BUFFERSIZE;

  } else
  {
    lRet = g_pcDeviceHandler->EnumerateChannels( ulBoard, ulChannel, ptChannelInfo);
  }

  return lRet;
}

/*! **************************************************************************
* Get/Return a memory pointer to the boards dual-port memory
*   \param hDriver       Driver handle, if NULL the driver will be temporarily opened by the API
*   \param ulBoardNumber The board number
*   \param ulCmd         Function command
*   \param pvMemoryInfo  Memory information structure
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xDriverMemoryPointer(HANDLE         hDriver,
                                   uint32_t  ulBoardNumber, 
                                   uint32_t  ulCmd,
                                   void*          pvMemoryInfo)
{
  UNREFERENCED_PARAMETER( hDriver);
  UNREFERENCED_PARAMETER( ulBoardNumber);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( pvMemoryInfo);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*! **************************************************************************
* Gets the error description in english from error number
*   \param lError         Error number returned by driver function
*   \param szBuffer       Return buffer for error description
*   \param ulBufferLen    Length of the return buffer
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xDriverGetErrorDescription( int32_t          lError,
                                          char*         szBuffer, 
                                          uint32_t ulBufferLen)
{
  CHECK_POINTER(szBuffer);  

  int32_t lRet = CIFX_INVALID_PARAMETER;

  DWORD dwResult = FormatMessage(FORMAT_MESSAGE_FROM_HMODULE,
                                 g_hModule,
                                 lError,
                                 MAKELANGID(LANG_ENGLISH, SUBLANG_DEFAULT),
                                 szBuffer,
                                 ulBufferLen,
                                 NULL);

  if(0 != dwResult)
  {
    //strip unneccesary line feeds/CRs at end of string
    int iStrLen = 0;
    while( (iStrLen = (int)strlen(szBuffer)) > 0)
    {
      if( (szBuffer[iStrLen - 1] == '\r') ||
          (szBuffer[iStrLen - 1] == '\n') )
      {
        szBuffer[iStrLen - 1] = '\0';
      } else
      {
        break;
      }
    }
    lRet = CIFX_NO_ERROR;
  }
  return lRet;
}

/***************************************************************************
* 
*   
*   CHANNEL FUNCTIONS....................  
*
*
*
*
*
******************************************************************************/
/*! **************************************************************************
* Opens a channel by name (Name can be obtained when enumerating Channels)
*   \param hDriver    Driver handle
*   \param szBoard    Name of the board
*   \param ulChannel  Channel number
*   \param phChannel  Returned handle to the channel (Needed for all channel specific operations)
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelOpen(HANDLE         hDriver,
                           char*          szBoard, 
                           uint32_t  ulChannel, 
                           HANDLE*        phChannel)
{
  if(0 == g_lDriverOpenCnt)
    return CIFX_DRV_NOT_OPENED;

  CHECK_DRIVERHANDLE(hDriver);
  CHECK_POINTER(szBoard);
  CHECK_POINTER(phChannel);

  return g_pcDeviceHandler->OpenDeviceConnection( szBoard, 
                                                  ulChannel, 
                                                  phChannel);
}

/*! **************************************************************************
* Closes an open channel
*   \param hChannel Channel handle acquired by xChannelOpen
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelClose(HANDLE hChannel)
{
  CHECK_CHANNELHANDLE(hChannel);

  return g_pcDeviceHandler->CloseDeviceConnection( hChannel);
}

/*! **************************************************************************
* Reset the channel
*   \param hChannel     Channel handle acquired by xChannelOpen
*   \param ulResetMode  Cold/Warmstart selection
*   \param ulTimeout    Time in ms to wait for the reset
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelReset(HANDLE        hChannel,
                            uint32_t ulResetMode, 
                            uint32_t ulTimeout)
{
  CHECK_CHANNELHANDLE(hChannel);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  int32_t lRet = CIFX_NO_ERROR;

  /* Let the device handler handle the reset function */
  switch (ulResetMode)
  {
    case CIFX_SYSTEMSTART:
      lRet = g_pcDeviceHandler->HandleReset(eCHANNELRESET, pcChannel, ulTimeout);
      break;
    
    case CIFX_CHANNELINIT:
      lRet = g_pcDeviceHandler->HandleReset(eCHANNELINIT, pcChannel, ulTimeout);
      break;
    
    default:
      lRet = CIFX_INVALID_PARAMETER;
      break;
  }
  
  return lRet;
}

/*! **************************************************************************
* Returns Channel specific information (same information as from Channel enumeration)
*   \param hChannel       Channel handle acquired by xChannelOpen
*   \param ulSize         Size of the user buffer
*   \param pvChannelInfo  Returned channel specific information
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelInfo(HANDLE         hChannel,
                           uint32_t  ulSize, 
                           void*          pvChannelInfo)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(pvChannelInfo);

  CHANNEL_INFORMATION* ptChannelInfo  = (CHANNEL_INFORMATION*)pvChannelInfo;
  if( ulSize < (uint32_t)sizeof(*ptChannelInfo))
    return CIFX_INVALID_BUFFERSIZE;

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  int32_t lRet = CIFX_NO_ERROR;
  if( CIFX_NO_ERROR == (lRet = pcDataLayer->xChannelInfo(pcChannel->m_hChannel, 
                                                         ulSize,
                                                         pvChannelInfo)))
  {
    // Insert the correct device name
    std::string szInterfaceName;
    pcDevCon->m_pcEndpoint->m_pcInterface->GetInterfaceName(szInterfaceName);

    // Return the complete name for the user
    g_pcDeviceHandler->CreateInterfaceName(szInterfaceName, 
                                           pcDevCon->m_tBoardInfo.abBoardName, 
                                           sizeof(ptChannelInfo->abBoardName)+sizeof(ptChannelInfo->abBoardAlias), 
                                           ptChannelInfo->abBoardName);
  }

  return lRet;
}

/*! **************************************************************************
* Returns the Mailbox state from a specific channel
*   \param hChannel         Channel handle acquired by xChannelOpen
*   \param pulRecvPktCount  Number of Messages waiting in receive mailbox
*   \param pulSendMbxState  State of the Send Mailbox
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelGetMBXState(HANDLE          hChannel,
                                  uint32_t*  pulRecvPktCount, 
                                  uint32_t*  pulSendMbxState)
{
  CHECK_POINTER(hChannel);
  CHECK_POINTER(pulRecvPktCount);
  CHECK_POINTER(pulSendMbxState);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelGetMBXState(pcChannel->m_hChannel, 
                                          pulRecvPktCount,
                                          pulSendMbxState);
}

/*! **************************************************************************
* Inserts a packet into a channel mailbox
*   \param hChannel   Channel handle acquired by xChannelOpen
*   \param ptSendPkt  Packet to send to channel
*   \param ulTimeout  Time in ms to wait for card to accept the packet
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelPutPacket(HANDLE        hChannel,
                                CIFX_PACKET*  ptSendPkt, 
                                uint32_t ulTimeout)
{
  CHECK_POINTER(hChannel);
  CHECK_POINTER(ptSendPkt);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelPutPacket(pcChannel->m_hChannel, 
                                        ptSendPkt,
                                        ulTimeout);
}

/*! **************************************************************************
* Gets a packet from a channel mailbox
*   \param hChannel   Channel handle acquired by xChannelOpen
*   \param ulSize     Size of returned packet
*   \param ptRecvPkt  Returned packet
*   \param ulTimeout  Time in ms to wait for available packets
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelGetPacket(HANDLE        hChannel,
                                uint32_t ulSize, 
                                CIFX_PACKET*  ptRecvPkt, 
                                uint32_t ulTimeout)
{
  CHECK_POINTER(hChannel);
  CHECK_POINTER(ptRecvPkt);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelGetPacket(pcChannel->m_hChannel, 
                                        ulSize,
                                        ptRecvPkt,
                                        ulTimeout);
}

/*! **************************************************************************
* Read back a packet from a channel send mailbox
*   \param hChannel   Channel handle acquired by xChannelOpen
*   \param ulSize     Size of returned packet
*   \param ptRecvPkt  Returned packet
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelGetSendPacket(  HANDLE        hChannel, 
                                      uint32_t ulSize,
                                      CIFX_PACKET*  ptRecvPkt)
{
  CHECK_POINTER(hChannel);
  CHECK_POINTER(ptRecvPkt);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelGetSendPacket(pcChannel->m_hChannel, 
                                            ulSize,
                                            ptRecvPkt);
}

/*! **************************************************************************
* Trigger the channel watchdog
*   \param hChannel    Channel handle acquired by xChannelOpen
*   \param ulCmd       Watchdog command 
*   \param pulTrigger  Actual trigger value from hardware
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelWatchdog( HANDLE          hChannel,
                                uint32_t   ulCmd,
                                uint32_t*  pulTrigger)
{
  CHECK_POINTER(hChannel);
  CHECK_POINTER(pulTrigger);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelWatchdog(pcChannel->m_hChannel, 
                                       ulCmd,
                                       pulTrigger);
}

/*! **************************************************************************
* Set/Get Host state 
*   \param hChannel   Channel handle acquired by xChannelOpen
*   \param ulCmd      Function command
*   \param pulState   Actual state returned by the get command
*   \param ulTimeout  Time in ms to wait for the defined state
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelHostState(  HANDLE          hChannel,
                                  uint32_t   ulCmd,
                                  uint32_t*  pulState, 
                                  uint32_t   ulTimeout)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(pulState);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelHostState(pcChannel->m_hChannel, 
                                        ulCmd,
                                        pulState,
                                        ulTimeout);
} 

/*! **************************************************************************
* Set/Get Configuration Lock state
*   \param hChannel   Channel handle acquired by xChannelOpen
*   \param ulCmd      Function command
*   \param pulState   Actual state returned by the get command
*   \param ulTimeout  Time in ms to wait for the defined state
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelConfigLock( HANDLE          hChannel,
                                  uint32_t   ulCmd,
                                  uint32_t*  pulState,
                                  uint32_t   ulTimeout)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(pulState);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelConfigLock(pcChannel->m_hChannel, 
                                         ulCmd,
                                         pulState,
                                         ulTimeout);
}

/*! **************************************************************************
* Set/Get Bus state
*   \param hChannel   Channel handle acquired by xChannelOpen
*   \param ulCmd      Function command
*   \param pulState   Actual state returned by the get command
*   \param ulTimeout  Time in ms to wait for the defined state
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelBusState( HANDLE          hChannel,
                                uint32_t   ulCmd,
                                uint32_t*  pulState, 
                                uint32_t   ulTimeout)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(pulState);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelBusState(pcChannel->m_hChannel, 
                                       ulCmd,
                                       pulState,
                                       ulTimeout);
}

/*! **************************************************************************
* Set/Get DMA state (not supported)
*   \param hChannel   Channel handle acquired by xChannelOpen
*   \param ulCmd      Function command
*   \param pulState   Actual state returned by the get command
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelDMAState(HANDLE hChannel, uint32_t ulCmd, uint32_t* pulState)
{
  UNREFERENCED_PARAMETER(hChannel);
  UNREFERENCED_PARAMETER(ulCmd);
  UNREFERENCED_PARAMETER(pulState);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*****************************************************************************/
/*! Register notification callbacks (not available on netX transport)
*   \param hChannel         Channel handle
*   \param ulNotification   Notification command
*   \param pfnCallback      Callback function
*   \param pvUser           User parameter for Callback function
*   \return CIFX_FUNCTION_NOT_AVAILABLE                                      */
/*****************************************************************************/
int32_t APIENTRY xChannelRegisterNotification(CIFXHANDLE  hChannel, uint32_t ulNotification, PFN_NOTIFY_CALLBACK  pfnCallback, void* pvUser)
{
  UNREFERENCED_PARAMETER(hChannel);
  UNREFERENCED_PARAMETER(ulNotification);
  UNREFERENCED_PARAMETER(pfnCallback);
  UNREFERENCED_PARAMETER(pvUser);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*****************************************************************************/
/*! Unregister notification callbacks (not available on netX transport)
*   \param hChannel         Channel handle
*   \param ulNotification   Notification command
*   \return CIFX_FUNCTION_NOT_AVAILABLE                                      */
/*****************************************************************************/
int32_t APIENTRY xChannelUnregisterNotification( CIFXHANDLE  hChannel, uint32_t ulNotification)
{
  UNREFERENCED_PARAMETER(hChannel);
  UNREFERENCED_PARAMETER(ulNotification);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*****************************************************************************/
/*! Wait / Set synch more (not available on netX transport)
*   \param hChannel         Channel handle
*   \param ulCmd            Synch command
*   \param ulTimeout        Timeout in ms
*   \param pulErrorCount    returned error counter
*   \return CIFX_FUNCTION_NOT_AVAILABLE                                      */
/*****************************************************************************/
int32_t APIENTRY xChannelSyncState(CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t ulTimeout, uint32_t* pulErrorCount)
{
  UNREFERENCED_PARAMETER(hChannel);
  UNREFERENCED_PARAMETER(ulCmd);
  UNREFERENCED_PARAMETER(ulTimeout);
  UNREFERENCED_PARAMETER(pulErrorCount);

  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*! **************************************************************************
* Get area information about IO Block
*   \param hChannel     Channel handle acquired by xChannelOpen
*   \param ulCmd        CIFX_IO_INPUT_AREA/CIFX_IO_OUTPUT_AREA
*   \param ulAreaNumber Area number to get data from
*   \param ulSize       Size of passed pointer
*   \param pvData       Pointer to CHANNEL_IO_INFORMATION structure
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelIOInfo( CIFXHANDLE    hChannel, 
                              uint32_t ulCmd, 
                              uint32_t ulAreaNumber, 
                              uint32_t ulSize,
                              void*         pvData)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(pvData);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelIOInfo(pcChannel->m_hChannel, 
                                     ulCmd,
                                     ulAreaNumber,
                                     ulSize,
                                     pvData);
}

/*! **************************************************************************
* Reads the Input data from the channel
*   \param hChannel     Channel handle acquired by xChannelOpen
*   \param ulAreaNumber Area number
*   \param ulOffset     Data offset in Input area
*   \param ulDataLen    Length of data to read
*   \param pvData       Buffer to place returned data
*   \param ulTimeout    Time to wait for Send confirmation [ms]
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelIORead( HANDLE        hChannel, 
                              uint32_t ulAreaNumber, 
                              uint32_t ulOffset, 
                              uint32_t ulDataLen, 
                              void*         pvData,
                              uint32_t ulTimeout)
{
  CHECK_POINTER(hChannel);
  CHECK_POINTER(pvData);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelIORead(pcChannel->m_hChannel, 
                                     ulAreaNumber,
                                     ulOffset,
                                     ulDataLen,
                                     pvData,
                                     ulTimeout);
}

/*! **************************************************************************
* Writes the Output data to the channel
*   \param hChannel     Channel handle acquired by xChannelOpen
*   \param ulAreaNumber Area number
*   \param ulOffset     Data offset in Output area
*   \param ulDataLen    Length of data to send
*   \param pvWriteData  Buffer containing send data
*   \param ulTimeout    Time to wait for Send confirmation [ms]
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelIOWrite(  HANDLE        hChannel, 
                                uint32_t ulAreaNumber, 
                                uint32_t ulOffset, 
                                uint32_t ulDataLen, 
                                void*         pvWriteData,
                                uint32_t ulTimeout)
{
  CHECK_POINTER(hChannel);
  CHECK_POINTER(pvWriteData);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelIOWrite(pcChannel->m_hChannel, 
                                      ulAreaNumber,
                                      ulOffset,
                                      ulDataLen,
                                      pvWriteData,
                                      ulTimeout);
}

/*! **************************************************************************
* Read back Send Data Area from channel
*   \param hChannel     Channel handle acquired by xChannelOpen
*   \param ulAreaNumber Area number
*   \param ulOffset     Data start offset
*   \param ulDataLen    Data length to read
*   \param pvData       Data buffer
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelIOReadSendData( HANDLE        hChannel,
                                      uint32_t ulAreaNumber, 
                                      uint32_t ulOffset, 
                                      uint32_t ulDataLen, 
                                      void*         pvData)
{
  CHECK_POINTER(hChannel);
  CHECK_POINTER(pvData);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelIOReadSendData(pcChannel->m_hChannel, 
                                             ulAreaNumber,
                                             ulOffset,
                                             ulDataLen,
                                             pvData);
}

/*! **************************************************************************
* Acces the channels control block
*   \param hChannel     Channel handle acquired by xChannelOpen
*   \param ulCmd        Command
*   \param ulOffset     Data start offset
*   \param ulDataLen    Data length
*   \param pvData       Data buffer
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelControlBlock( HANDLE        hChannel, 
                                    uint32_t ulCmd,
                                    uint32_t ulOffset, 
                                    uint32_t ulDataLen, 
                                    void*         pvData)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(pvData);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelControlBlock(pcChannel->m_hChannel, 
                                           ulCmd,
                                           ulOffset,
                                           ulDataLen,
                                           pvData);
}

/*! **************************************************************************
* Acces the channels common status block
*   \param hChannel     Channel handle acquired by xChannelOpen
*   \param ulCmd        Command
*   \param ulOffset     Data start offset
*   \param ulDataLen    Data length
*   \param pvData       Data buffer
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelCommonStatusBlock( HANDLE        hChannel, 
                                         uint32_t ulCmd,
                                         uint32_t ulOffset, 
                                         uint32_t ulDataLen, 
                                         void*         pvData)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(pvData);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelCommonStatusBlock(pcChannel->m_hChannel, 
                                                ulCmd,
                                                ulOffset,
                                                ulDataLen,
                                                pvData);
}                                         

/*! **************************************************************************
* Acces the channels common status block
*   \param hChannel     Channel handle acquired by xChannelOpen
*   \param ulAreaNumber I/O Area number
*   \param ulCmd        Command
*   \param ulOffset     Data start offset
*   \param ulDataLen    Data length
*   \param pvData       Data buffer
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelUserBlock(  HANDLE        hChannel, 
                                  uint32_t ulAreaNumber,
                                  uint32_t ulCmd,
                                  uint32_t ulOffset, 
                                  uint32_t ulDataLen, 
                                  void*         pvData)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( ulOffset);
  UNREFERENCED_PARAMETER( ulDataLen);
  UNREFERENCED_PARAMETER( pvData);
    
  return CIFX_FUNCTION_NOT_AVAILABLE;
}                                         

/*! **************************************************************************
* Acces the channels extended status block
*   \param hChannel     Channel handle acquired by xChannelOpen
*   \param ulCmd        Command
*   \param ulOffset     Data start offset
*   \param ulDataLen    Data length
*   \param pvData       Data buffer
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelExtendedStatusBlock( HANDLE        hChannel, 
                                           uint32_t ulCmd,
                                           uint32_t ulOffset, 
                                           uint32_t ulDataLen, 
                                           void*         pvData)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(pvData);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelExtendedStatusBlock(pcChannel->m_hChannel, 
                                                  ulCmd,
                                                  ulOffset,
                                                  ulDataLen,
                                                  pvData);
}                                         

/*! **************************************************************************
* Get a Pointer to the I/O Area
*   \param hChannel       Channel handle acquired by xChannelOpen
*   \param ulCmd          Command
*   \param pvMemoryInfo   Returned I/O Area pointer
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelPLCMemoryPtr(HANDLE hChannel, uint32_t ulCmd, void* pvMemoryInfo)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( pvMemoryInfo);
    
  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*! **************************************************************************
* Checks if Input Data area has been updated and is ready for reading
*   \param hChannel       Channel handle acquired by xChannelOpen
*   \param ulAreaNumber   Area Number
*   \param pulReadState   State of the input area
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelPLCIsReadReady(HANDLE hChannel, uint32_t ulAreaNumber, uint32_t* pulReadState)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
  UNREFERENCED_PARAMETER( pulReadState);
    
  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*! **************************************************************************
* Checks if Output Data area has been read by the firmware
*   \param hChannel       Channel handle acquired by xChannelOpen
*   \param ulAreaNumber   Area number
*   \param pulWriteState  State of the output area
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelPLCIsWriteReady(HANDLE hChannel, uint32_t ulAreaNumber, uint32_t* pulWriteState)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
  UNREFERENCED_PARAMETER( pulWriteState);
    
  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*! **************************************************************************
* Requests Firmware to read the new send data
*   \param hChannel       Channel handle acquired by xChannelOpen
*   \param ulAreaNumber   Area number
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelPLCActivateWrite(HANDLE hChannel, uint32_t ulAreaNumber)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
    
  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*! **************************************************************************
* Requests Firmware to update input data
*   \param hChannel       Channel handle acquired by xChannelOpen
*   \param ulAreaNumber   I/O Area number
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelPLCActivateRead(HANDLE hChannel, uint32_t ulAreaNumber)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
    
  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*! **************************************************************************
* Download a file (Firmware, Configuratio, etc) to the device
*   \param hChannel           Handle to the channel
*   \param ulMode             Download mode (DOWNLOAD_MODE_FIRMWARE, etc)
*   \param szFileName         Name of the file
*   \param pabFileData        Pointer to the file data
*   \param ulFileSize         Length of the file data
*   \param pfnCallback        Callback for progress indication (NULL for no callback)
*   \param pfnRecvPktCallback Callback pointer for unhandled receive packets
*   \param pvUser             User parameter on callback. 
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xChannelDownload( HANDLE                hChannel,
                               uint32_t          ulMode, 
                               char*                  szFileName, 
                               uint8_t*         pabFileData, 
                               uint32_t          ulFileSize, 
                               PFN_PROGRESS_CALLBACK  pfnCallback,
                               PFN_RECV_PKT_CALLBACK  pfnRecvPktCallback, 
                               void*                  pvUser)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(szFileName);
  CHECK_POINTER(pabFileData);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelDownload(pcChannel->m_hChannel, 
                                       ulMode,
                                       szFileName,
                                       pabFileData,
                                       ulFileSize,
                                       pfnCallback,
                                       pfnRecvPktCallback,
                                       pvUser);
}

/*****************************************************************************/
/*! Starts directory enumeration on the given channel
*   \param hChannel           Handle to the channel
*   \param ptDirectoryInfo    Pointer to enumeration result. 
*                             (Will be initialized inside function)
*   \param pfnRecvPktCallback Callback pointer for unhandled receive packets
*   \param pvUser             User parameter on callback. 
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t APIENTRY xChannelFindFirstFile(  CIFXHANDLE            hChannel,
                                      CIFX_DIRECTORYENTRY*  ptDirectoryInfo,
                                      PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                      void*                 pvUser)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(ptDirectoryInfo);  

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelFindFirstFile(pcChannel->m_hChannel, 
                                            ptDirectoryInfo,
                                            pfnRecvPktCallback,
                                            pvUser);
}

/*****************************************************************************/
/*! Enumerate next entry in directoy on the given channel
*   \param hChannel           Handle to the channel
*   \param ptDirectoryInfo    Pointer to enumeration result. 
*   \param pfnRecvPktCallback Callback pointer for unhandled receive packets
*   \param pvUser             User parameter on callback. 
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t APIENTRY xChannelFindNextFile( CIFXHANDLE             hChannel,
                                    CIFX_DIRECTORYENTRY*   ptDirectoryInfo,
                                    PFN_RECV_PKT_CALLBACK  pfnRecvPktCallback,
                                    void*                  pvUser)
{
  CHECK_CHANNELHANDLE(hChannel);
  CHECK_POINTER(ptDirectoryInfo);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelFindNextFile(pcChannel->m_hChannel, 
                                           ptDirectoryInfo,
                                           pfnRecvPktCallback,
                                           pvUser);
}

/*****************************************************************************/
/*! Uploads a file via Communication channel
*   \param hChannel           Handle to the Channel
*   \param ulMode             Transfer Mode
*   \param szFileName         Filename to upload
*   \param pabFileData        Pointer to buffer receiving upload
*   \param pulFileSize        [in]Length of buffer, [out] Bytes copied to buffer
*   \param pfnCallback        Callback pointer for progress
*   \param pfnRecvPktCallback Callback pointer for unhandled receive packets
*   \param pvUser             User parameter on callback. 
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t APIENTRY xChannelUpload( CIFXHANDLE             hChannel,
                              uint32_t          ulMode, 
                              char*                  szFileName, 
                              uint8_t*         pabFileData, 
                              uint32_t*         pulFileSize, 
                              PFN_PROGRESS_CALLBACK  pfnCallback, 
                              PFN_RECV_PKT_CALLBACK  pfnRecvPktCallback, 
                              void*                  pvUser)
{ 
  CHECK_CHANNELHANDLE(hChannel);  
  CHECK_POINTER(szFileName);
  CHECK_POINTER(pabFileData);
  CHECK_POINTER(pulFileSize);

  CChannel* pcChannel = (CChannel*)hChannel;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xChannelUpload(pcChannel->m_hChannel, 
                                     ulMode,
                                     szFileName,
                                     pabFileData,
                                     pulFileSize,
                                     pfnCallback,
                                     pfnRecvPktCallback,
                                     pvUser);
}

/***************************************************************************
* 
*   
*   SYSDEVICE FUNCTIONS....................  
*
*
*
*
*
******************************************************************************/

/*! **************************************************************************
* Opens the System device on the given board
*   \param hDriver      Driver handle
*   \param szBoard      Name of the board to open
*   \param phSysdevice  Returned handle to the System device area
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xSysdeviceOpen( HANDLE  hDriver,
                              char*   szBoard, 
                              HANDLE* phSysdevice)
{
  if(0 == g_lDriverOpenCnt)
    return CIFX_DRV_NOT_OPENED;

  CHECK_DRIVERHANDLE(hDriver);
  CHECK_POINTER(szBoard);
  CHECK_POINTER(phSysdevice);

  // OpenDeviceConnection handles board name and human readable name of a device
  return g_pcDeviceHandler->OpenDeviceConnection( szBoard,
                                                  CIFX_SYSTEM_DEVICE, 
                                                  phSysdevice);
}

/*! **************************************************************************
* Closed an open System device
*   \param hSysdevice  Handle to the System device to close
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xSysdeviceClose(HANDLE hSysdevice)
{
  CHECK_SYSDEVICEHANDLE(hSysdevice);

  return g_pcDeviceHandler->CloseDeviceConnection( hSysdevice);
}

/*! **************************************************************************
* Gets the information of a system device
*   \param hSysdevice   Handle to the system device
*   \param ulCmd        Command
*   \param ulSize       Size of the passed structure
*   \param pvInfo       Pointer to the structure for returned data
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xSysdeviceInfo( HANDLE        hSysdevice,
                              uint32_t ulCmd, 
                              uint32_t ulSize, 
                              void*         pvInfo)
{
  CHECK_SYSDEVICEHANDLE(hSysdevice);
  CHECK_POINTER(pvInfo);

  CChannel* pcChannel = (CChannel*)hSysdevice;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xSysdeviceInfo(pcChannel->m_hChannel, ulCmd, ulSize, pvInfo);
}

/*! **************************************************************************
* Gets the Mailbox state of an open system device
*   \param hSysdevice      Handle to the System device
*   \param pulRecvPktCount Number of packets in receive mailbox
*   \param pulSendPktCount Number of packets the application is able to send at once
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xSysdeviceGetMBXState(  HANDLE          hSysdevice,
                                      uint32_t*  pulRecvPktCount,
                                      uint32_t*  pulSendPktCount)
{
  CHECK_POINTER(hSysdevice);
  CHECK_POINTER(pulRecvPktCount);
  CHECK_POINTER(pulSendPktCount);

  CChannel* pcChannel = (CChannel*)hSysdevice;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xSysdeviceGetMBXState(pcChannel->m_hChannel, 
                                            pulRecvPktCount, 
                                            pulSendPktCount);
}

/*! **************************************************************************
* Inserts a packet into the System Mailbox
*   \param hSysdevice      Handle to the System device
*   \param ptSendPkt       Packet to send to device
*   \param ulTimeout       maximum time to wait for packet to be accepted by device (in ms)
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xSysdevicePutPacket(  HANDLE        hSysdevice, 
                                    CIFX_PACKET*  ptSendPkt, 
                                    uint32_t ulTimeout)
{
  CHECK_POINTER(hSysdevice);
  CHECK_POINTER(ptSendPkt);

  CChannel* pcChannel = (CChannel*)hSysdevice;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xSysdevicePutPacket(pcChannel->m_hChannel, 
                                          ptSendPkt, 
                                          ulTimeout);
}

/*! **************************************************************************
* Retrieves a packet from the System Mailbox
*   \param hSysdevice      Handle to the System device
*   \param ulSize          Size of the buffer to retrieve the packet
*   \param ptRecvPkt       Pointer to buffer for received packet
*   \param ulTimeout       maximum time to wait for packet to be delivered by device (in ms)
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xSysdeviceGetPacket(  HANDLE        hSysdevice, 
                                    uint32_t ulSize, 
                                    CIFX_PACKET*  ptRecvPkt,
                                    uint32_t ulTimeout)
{
  CHECK_POINTER(hSysdevice);
  CHECK_POINTER(ptRecvPkt);

  CChannel* pcChannel = (CChannel*)hSysdevice;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xSysdeviceGetPacket(pcChannel->m_hChannel, 
                                          ulSize,
                                          ptRecvPkt, 
                                          ulTimeout);
}

/*! **************************************************************************
* Reset the whole device (coldstart)
*   \param hSysdevice   Sysdevice handle acquired by xSysdeviceOpen
*   \param ulTimeout    Time in ms to wait for the reset
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xSysdeviceReset(HANDLE        hSysdevice, 
                              uint32_t ulTimeout)
{
  CHECK_SYSDEVICEHANDLE(hSysdevice);

  CChannel* pcChannel = (CChannel*)hSysdevice;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  /* Let the device handler handle the reset function */
  int32_t lRet = g_pcDeviceHandler->HandleReset( eSYSTEMRESET,
                                              pcChannel,
                                              ulTimeout);
  return lRet;
}

/*! **************************************************************************
* Download a file (Firmware, Configuratio, etc) to the device
*   \param hSysdevice         Handle to the system device
*   \param ulChannel          Channel number to load the file to
*   \param ulMode             Download mode (DOWNLOAD_MODE_FIRMWARE, etc)
*   \param szFileName         Name of the file
*   \param pabFileData        Pointer to the file data
*   \param ulFileSize         Length of the file data
*   \param pfnCallback        Callback for progress indication (NULL for no callback)
*   \param pfnRecvPktCallback Callback pointer for unhandled receive packets
*   \param pvUser             User parameter on callback. 
*   \return CIFX_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY xSysdeviceDownload( HANDLE                hSysdevice,
                                  uint32_t         ulChannel,    
                                  uint32_t         ulMode, 
                                  char*                 szFileName,  
                                  uint8_t*        pabFileData, 
                                  uint32_t         ulFileSize, 
                                  PFN_PROGRESS_CALLBACK pfnCallback,
                                  PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                  void*                 pvUser)
{
  CHECK_SYSDEVICEHANDLE(hSysdevice);
  CHECK_POINTER(szFileName);
  CHECK_POINTER(pabFileData);

  CChannel* pcChannel = (CChannel*)hSysdevice;
  
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xSysdeviceDownload(pcChannel->m_hChannel, 
                                         ulChannel,
                                         ulMode,
                                         szFileName,
                                         pabFileData,
                                         ulFileSize,
                                         pfnCallback,
                                         pfnRecvPktCallback,
                                         pvUser);
}

/*****************************************************************************/
/*! Starts directory enumeration on the given channel
*   \param hSysdevice         Handle to the channel
*   \param ulChannel          Channel number to get directory from
*   \param ptDirectoryInfo    Pointer to enumeration result. 
*                             (Will be initialized inside function)
*   \param pfnRecvPktCallback Callback pointer for unhandled receive packets
*   \param pvUser             User parameter on callback. 
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t APIENTRY xSysdeviceFindFirstFile(  CIFXHANDLE            hSysdevice,
                                        uint32_t         ulChannel,
                                        CIFX_DIRECTORYENTRY*  ptDirectoryInfo,
                                        PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                        void*                 pvUser)
{ 
  CHECK_SYSDEVICEHANDLE(hSysdevice);
  CHECK_POINTER(ptDirectoryInfo);

  CChannel* pcChannel = (CChannel*)hSysdevice;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xSysdeviceFindFirstFile(pcChannel->m_hChannel, 
                                              ulChannel,
                                              ptDirectoryInfo,
                                              pfnRecvPktCallback,
                                              pvUser);
}

/*****************************************************************************/
/*! Enumerate next entry in directoy on the given channel
*   \param hSysdevice         Handle to the system device
*   \param ulChannel          Channel number to get directory from
*   \param ptDirectoryInfo    Pointer to enumeration result. 
*   \param pfnRecvPktCallback Callback pointer for unhandled receive packets
*   \param pvUser             User parameter on callback. 
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t APIENTRY xSysdeviceFindNextFile( CIFXHANDLE            hSysdevice,
                                      uint32_t         ulChannel,
                                      CIFX_DIRECTORYENTRY*  ptDirectoryInfo,
                                      PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                      void*                 pvUser)
{
  CHECK_SYSDEVICEHANDLE(hSysdevice);
  CHECK_POINTER(ptDirectoryInfo);

  CChannel* pcChannel = (CChannel*)hSysdevice;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xSysdeviceFindNextFile(pcChannel->m_hChannel, 
                                             ulChannel,
                                             ptDirectoryInfo,
                                             pfnRecvPktCallback,
                                             pvUser);
}

/*****************************************************************************/
/*! Uploads a file via system channel
*   \param hSysdevice         Handle to the System device
*   \param ulChannel          Channel number to get directory from
*   \param ulMode             Transfer Mode
*   \param szFileName         Filename to upload
*   \param pabFileData        Pointer to buffer receiving upload
*   \param pulFileSize        [in]Length of buffer, [out] Bytes copied to buffer
*   \param pfnCallback        Callback pointer for progress
*   \param pfnRecvPktCallback Callback pointer for unhandled receive packets
*   \param pvUser             User parameter on callback. 
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t APIENTRY xSysdeviceUpload( CIFXHANDLE            hSysdevice,
                                uint32_t         ulChannel, 
                                uint32_t         ulMode,
                                char*                 szFileName, 
                                uint8_t*        pabFileData, 
                                uint32_t*        pulFileSize, 
                                PFN_PROGRESS_CALLBACK pfnCallback,
                                PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, 
                                void*                 pvUser)
{
  CHECK_SYSDEVICEHANDLE(hSysdevice);
  CHECK_POINTER(szFileName);
  CHECK_POINTER(pabFileData);
  CHECK_POINTER(pulFileSize);

  CChannel* pcChannel = (CChannel*)hSysdevice;
  if( !CheckHandle(pcChannel))
    return CIFX_INVALID_CHANNEL; 

  CDevice*    pcDevCon    = pcChannel->m_pcDevice;
  CDataLayer* pcDataLayer = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;

  return pcDataLayer->xSysdeviceUpload(pcChannel->m_hChannel, 
                                       ulChannel,
                                       ulMode,
                                       szFileName,
                                       pabFileData,
                                       pulFileSize,
                                       pfnCallback,
                                       pfnRecvPktCallback, 
                                       pvUser);
}

/*****************************************************************************/
/*! Boot start reset to via system channel
*   \param hSysdevice Handle to system device
*   \param ulTimeout  Timeout to wait for card to finish reset
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t APIENTRY xSysdeviceBootstart(CIFXHANDLE hSysdevice, uint32_t ulTimeout)
{
  UNREFERENCED_PARAMETER( hSysdevice);
  UNREFERENCED_PARAMETER( ulTimeout);
    
  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/*****************************************************************************/
/*! Get/Return a memory pointer to an extended board memory if available
*   \param hSysdevice   Handle to system device
*   \param ulCmd        Command for get/free
*   \param ptExtMemInfo Pointer to a user buffer to return the information
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t APIENTRY xSysdeviceExtendedMemory(CIFXHANDLE hSysdevice, uint32_t ulCmd, CIFX_EXTENDED_MEMORY_INFORMATION* ptExtMemInfo)
{
  UNREFERENCED_PARAMETER( hSysdevice);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( ptExtMemInfo);
    
  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/***************************************************************************
* 
*   
*   netX API Global Functions....................  
*
*
*
*
*
******************************************************************************/

/*! **************************************************************************
* Read netX API information
*   \param ulInfoSize    Size of the user buffer
*   \param ptDrvInfo     Pointer to the user information buffer
*   \return NAXPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvGetInformation ( uint32_t ulInfoSize, PNXDRV_HW_INFORMATION ptDrvInfo)
{
  if( NULL == ptDrvInfo) 
    return NXAPI_INVALID_PARAMETER;
  
  if ( ulInfoSize < sizeof(*ptDrvInfo))
    return NXAPI_BUFFER_TOO_SHORT;

  CFileVersionInfo cVerInfo;
  
  if (cVerInfo.Open(g_hModule))
  {
    cVerInfo.QueryStringValue(VI_STR_FILEVERSION, ptDrvInfo->szVersion, sizeof(ptDrvInfo->szVersion)/sizeof(ptDrvInfo->szVersion[0]));
    cVerInfo.QueryStringValue(VI_STR_PRODUCTNAME, ptDrvInfo->szDriverName, sizeof(ptDrvInfo->szDriverName)/sizeof(ptDrvInfo->szDriverName[0]));
    cVerInfo.Close();
  }

  ptDrvInfo->ulDriverType         = s_ulDriverType;
  ptDrvInfo->ulDeviceClass        = RCX_HW_DEV_CLASS_UNDEFINED;
  ptDrvInfo->ulDriverRequirements = NXDRV_REQ_STARTUP_SW;

  return NXAPI_NO_ERROR;
}

/*! **************************************************************************
* Initialize API interface
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvInit ( void)
{
  return NXAPI_NO_ERROR;
}

/*! **************************************************************************
* Deinitialize API interface
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvExit ( void)
{
  // Clear device list
  g_cmNxDevInfoList.clear();

  return NXAPI_NO_ERROR;
}

/*! **************************************************************************
* Find available devices
*   \param  ulCmd         Command for the FindDevice function
*   \param  ulInfoSize    Size of the information structure 
*   \param  ptDeviceInfo  Information of the device 
*   \param  pulSearchIdx  Search index 
*   \return NAXPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvFindDevice(uint32_t             ulCmd,
                                 uint32_t             ulInfoSize, 
                                 NXDRV_DEVICE_INFORMATION* ptDeviceInfo,
                                 uint32_t*            pulSearchIdx)
{
  // Check incomming parameters 
  if( (NULL == ptDeviceInfo) ||
      (NULL == pulSearchIdx))
    return NXAPI_INVALID_PARAMETER;

  if(ulInfoSize < (uint32_t)sizeof(NXDRV_DEVICE_INFORMATION))
    return NXAPI_BUFFER_TOO_SHORT;

  switch (ulCmd)
  {
    case NXDRV_FIND_FIRST:
      *pulSearchIdx = 0;
      break;

    case NXDRV_FIND_NEXT:
      break;

    default:
      return NXAPI_UNKOWN_COMMAND;
  }

  int32_t          lRet        = NXAPI_NO_ENTRIES;
  uint32_t ulDeviceIdx = 0;

  /* In earlier versions, the device list was created during nxDrvInit(). */
  /* But this takes too long, depending on how many interfaces must be searched. */
  /* New application using the nxDrvBrowseDevices() functions which supports a progress call back.*/
  /* For older application we now to call nxDrvBrowseDevices() before looking into the device list.*/
  /* Otherwise the device list ist always empty!!! */
  if( NXDRV_FIND_FIRST == ulCmd )
    lRet = nxDrvBrowseDevices( NULL, NULL);

  /* Search for device in list */
  for (NXAPI_DEVICE_INFO_LIST::iterator iterDevice = g_cmNxDevInfoList.begin(); iterDevice != g_cmNxDevInfoList.end(); iterDevice++)
  {
    if (*pulSearchIdx == ulDeviceIdx)
    {
      *ptDeviceInfo = iterDevice->second;
      *pulSearchIdx = (*pulSearchIdx) + 1;
      lRet = NXAPI_NO_ERROR;
      break;
    }
    ulDeviceIdx++;
  }

  return lRet;
}

/*! **************************************************************************
* Enumerate over all available Boards
*   \param pfnCallback  Progress callback
*   \param pvUser       User pointer for progress callback
*   \return CIFX_NO_ERROR on success, CIFX_NO_MORE_ENTRIES when ulBoard is unknown
******************************************************************************/
int32_t APIENTRY nxDrvBrowseDevices ( PFN_NXAPI_BROWSE_CALLBACK pfnCallback, 
                                      void*                     pvUser)
{
  int32_t                  lRet        = CIFX_NO_ERROR;
  HANDLE                hDriver     = 0; 

  g_cmNxDevInfoList.clear();

  if (CIFX_NO_ERROR == (lRet = xDriverOpen(&hDriver))) 
  {
    std::vector<BOARD_INFORMATION> cvBoardList;

    if( CIFX_NO_ERROR == (lRet = g_pcDeviceHandler->BrowseDevices( cvBoardList, pfnCallback, pvUser)))
    {
      for ( std::vector<BOARD_INFORMATION>::iterator iter = cvBoardList.begin(); iter != cvBoardList.end(); iter++)
      {
        /* Create list entry containing identifier, name and system channel information */
        NXDRV_DEVICE_INFORMATION tDevListEntry = {0};
        tDevListEntry.hDevice = (HANDLE) ( g_cmNxDevInfoList.size() + 1 );
        strncpy( tDevListEntry.szDeviceName, (*iter).abBoardName, sizeof((*iter).abBoardName)+sizeof((*iter).abBoardAlias));
        tDevListEntry.tSystemInfoBlock = (*iter).tSystemInfo;
        g_cmNxDevInfoList.insert(make_pair( tDevListEntry.hDevice, tDevListEntry));
      }
    }

    xDriverClose(hDriver);
  }

  return lRet;
}

/*! **************************************************************************
* Download files
*   \param  hDevice       Handle to a specific device 
*   \param  ulChannel     Number of specific channel, where the file should be downloaded
*   \param  ulCmd         Command for the download function (How to handle download)
*   \param  ulFileSize    Size of file 
*   \param  pszFileName   Name of file which should be downloaded
*   \param  pabFileData   Pointer reference to file data
*   \param  pvUser        User pointer 
*   \param  pfnCallback   Callback function pointer, for status information
*   \return NAXPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvDownload ( CIFXHANDLE      hDevice,      // This is the board
                                 uint32_t        ulChannel,
                                 uint32_t        ulCmd,        // Firmware/Config/Warmstart
                                 uint32_t        ulFileSize, 
                                 char*           pszFileName, 
                                 uint8_t*  pabFileData,
                                 void*           pvUser,
                                 PFN_NXAPI_PROGRESS_CALLBACK pfnCallback)
{
  // Check input values 
  if( (NULL == hDevice)                         ||
      (CIFX_MAX_NUMBER_OF_CHANNELS < ulChannel) ||                // check numerb of channel    
      (0 == ulFileSize)                         ||                // check size of file         
      (NULL == pszFileName)                     ||                // check pointer of filename  
      (NULL == pabFileData)                        )              // check pointer of data      
    return NXAPI_INVALID_PARAMETER;

  // Find device handle
  NXAPI_DEVICE_INFO_LIST::iterator iterDevice = g_cmNxDevInfoList.begin();
  iterDevice = g_cmNxDevInfoList.find(hDevice);
  if(iterDevice == g_cmNxDevInfoList.end())
    return NXAPI_DRIVER_NO_DEVICE_FOUND;
  
  HANDLE        hDriver         = 0;                        // Driver handle
  int32_t          lRet            = NXAPI_NO_ERROR;           // return errorcode 
  uint32_t ulDownloadMode  = 0;                        // new download command 
  HANDLE        hChannel        = NULL;                     // handle of channel 

  lRet = xDriverOpen(&hDriver);
  if( CIFX_NO_ERROR != lRet)
  {
    lRet = NXAPI_DRIVER_NOT_INITIALIZED;
  }else
  {
    // Check type of download 
    switch(ulCmd)
    {
      case NXAPI_CMD_FIRMWARE:
      {
        // Download firmware 
        ulDownloadMode = DOWNLOAD_MODE_FIRMWARE;
      }
      break;

      case NXAPI_CMD_CONFIGURATION:
      {
        // Download configuration 
        ulDownloadMode = DOWNLOAD_MODE_CONFIG;
      }
      break;

      case NXAPI_CMD_WARMSTART:
      case NXAPI_CMD_BOOTLOADER:
      default:
      {
        // Set unknown command 
        lRet = NXAPI_UNKOWN_COMMAND;
      }
      break;
    }

    if(NXAPI_NO_ERROR == lRet)
    {
      // Create a file name if full path is passed in pszFileName
      char szFileName[MAX_PATH] = {0};
      char szFileExt[MAX_PATH]  = {0};
      _splitpath( pszFileName,   // Full file name
                  NULL,          // Drive
                  NULL,          // Directory path
                  szFileName,    // File name
                  szFileExt);
      strcat( szFileName, szFileExt);
      
      // Check channel 
      if(CIFX_SYSTEM_DEVICE != ulChannel)
      {
        // xChannelDownload 
        if(CIFX_NO_ERROR == (lRet =  xChannelOpen(g_pcDeviceHandler, iterDevice->second.szDeviceName, ulChannel, &hChannel)))
        {
          //CALLBACKWRAPPER_T tCBWrapper  = {0};
          //tCBWrapper.pvUser             = pvUser;
          //tCBWrapper.pfnCallback        = pfnCallback;
          // Download on xChannel
          lRet =  xChannelDownload( hChannel,            // Channel
                                    ulDownloadMode,      // Download Mode
                                    szFileName,          // File name
                                    pabFileData,         // File data
                                    ulFileSize,          // Size
            (PFN_PROGRESS_CALLBACK) pfnCallback,         // Driver download callback
                                    NULL,                // Receive packet callback
                                    pvUser);             // pv User
          // Close channel 
          xChannelClose(hChannel);
        }
      } else 
      {
      //  xSysdeviceDownload 
        HANDLE hSysDevice = NULL;
        if(CIFX_NO_ERROR == (lRet = xSysdeviceOpen(g_pcDeviceHandler, iterDevice->second.szDeviceName, &hSysDevice)))
        {
          // Download on xSysdevice 
          lRet = xSysdeviceDownload(hSysDevice,
                                    ulChannel,
                                    ulDownloadMode,
                                    szFileName,
                                    pabFileData,
                                    ulFileSize,
                                    (PFN_PROGRESS_CALLBACK) pfnCallback,
                                    NULL,
                                    pvUser);
          // Close sysdevice 
          xSysdeviceClose(hSysDevice);
        }
      }
    }
    
    // Close the driver
    xDriverClose(hDriver);
  }

  return lRet;
}

/*! **************************************************************************
* Start a device
*   \param  hDevice       Handle of a specific device 
*   \param  ulChannel     Number of channel
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvStart ( HANDLE hDevice, uint32_t ulChannel)
{
  UNREFERENCED_PARAMETER( ulChannel);
  
  NXAPI_DEVICE_INFO_LIST::iterator iterDevice = g_cmNxDevInfoList.begin();

  // Check device handle (identifer)
  if(NULL == hDevice) 
  {
    return NXAPI_INVALID_PARAMETER;
  } else
  {
    iterDevice = g_cmNxDevInfoList.find(hDevice);
    if(iterDevice == g_cmNxDevInfoList.end())
      return NXAPI_DRIVER_NO_DEVICE_FOUND;
  }
  
  return xDriverRestartDevice(NULL, iterDevice->second.szDeviceName, NULL); 
}

/*! **************************************************************************
* Read the configuration information
*   \param  hDevice           Handle of a specific channnel
*   \param  ulCmd             Command for the config function 
*   \param  lChannel          Number of channel 
*   \param  ulCFGSearchIndex  Search index 
*   \param  ulBufferSize      Size of buffer
*   \param  ptCfgData         Pointer for config data 
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvGetConfigInfo ( CIFXHANDLE     hDevice,
                                      uint32_t       ulCmd, 
                                      int32_t        lChannel, 
                                      uint32_t       ulCFGSearchIndex, 
                                      uint32_t       ulBufferSize, 
                                      NXDRV_DRIVER_CFG_DATA_INFO* ptCfgData)
{
  UNREFERENCED_PARAMETER( hDevice);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( lChannel);
  UNREFERENCED_PARAMETER( ulCFGSearchIndex);
  UNREFERENCED_PARAMETER( ulBufferSize);
  UNREFERENCED_PARAMETER( ptCfgData);

  return NXAPI_UNKOWN_COMMAND;
}

/*! **************************************************************************
* Read a device configuration value
*   \param  hDevice       Handle of specific device 
*   \param  ulCmd         Command for the get configuration function 
*   \param  lChannel      Number of channel
*   \param  ptCfgData     Reference of config information structure
*   \param  ulBufferSize  Size of data buffer 
*   \param  pvData        Reference of data buffer 
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvGetConfig ( CIFXHANDLE hDevice, uint32_t ulCmd, int32_t lChannel, NXDRV_DRIVER_CFG_DATA_INFO* ptCfgData, uint32_t ulBufferSize, void* pvData)
{
  UNREFERENCED_PARAMETER( hDevice);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( lChannel);
  UNREFERENCED_PARAMETER( ptCfgData);
  UNREFERENCED_PARAMETER( ulBufferSize);
  UNREFERENCED_PARAMETER( pvData);

  return NXAPI_UNKOWN_COMMAND;
}

/*! **************************************************************************
* Write a device configuration value
*   \param  hDevice       Reference of data buffer 
*   \param  ulCmd         Command for the Set configuration function 
*   \param  lChannel      Number of channel
*   \param  ptCfgData     Reference of config information structure
*   \param  ulBufferSize  Size of data buffer 
*   \param  pvData        Reference of data buffer 
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvSetConfig ( HANDLE hDevice, uint32_t ulCmd, int32_t lChannel, NXDRV_DRIVER_CFG_DATA_INFO* ptCfgData,  uint32_t ulBufferSize, void* pvData)
{
  UNREFERENCED_PARAMETER( hDevice);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( lChannel);
  UNREFERENCED_PARAMETER( ptCfgData);
  UNREFERENCED_PARAMETER( ulBufferSize);
  UNREFERENCED_PARAMETER( pvData);

  return NXAPI_UNKOWN_COMMAND;
}

/*! **************************************************************************
* Delete the device configuration
*   \param  hDevice       Reference of data buffer 
*   \param  ulCmd         Command for the delete configuration function 
*   \param  lChannel      Number of channel
*   \param  ptCfgData     Reference of config information structure
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxDrvDeleteConfig ( HANDLE hDevice, uint32_t ulCmd, int32_t lChannel, NXDRV_DRIVER_CFG_DATA_INFO* ptCfgData)
{
  return NXAPI_UNKOWN_COMMAND;

  UNREFERENCED_PARAMETER( hDevice);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( lChannel);
  UNREFERENCED_PARAMETER( ptCfgData);
}

/*! **************************************************************************
* Enumerate Connectors
*   \param  ulConnectorIdx  Connector index
*   \param  ulSize          Size of the user buffer
*   \param  pvConnectorInfo Returned connector information
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxConEnumerate ( uint32_t ulConnectorIdx, 
                                  uint32_t ulSize,
                                  void*    pvConnectorInfo)
{
  int32_t lRet = NXAPI_NO_ERROR;

  if(sizeof(NXCON_CONNECTOR_INFO_T) > ulSize)
  {
    lRet = NXAPI_BUFFER_TOO_SHORT;

  // Return information of the first found connector 
  } else if( (g_cvConnectorDllList.empty())                 ||
             (g_cvConnectorDllList.size() <= ulConnectorIdx)  )
  {
    lRet = NXAPI_NO_ENTRIES;

  } else
  {
    // Handle information of the connector 
    PNXCON_CONNECTOR_INFO_T         ptConnector   = (PNXCON_CONNECTOR_INFO_T) pvConnectorInfo;
    NX_CONNECTOR_DLL_LIST::iterator iterConnector = g_cvConnectorDllList.begin();
    uint32_t                   ulSearchIdx   = 0;

    while (iterConnector != g_cvConnectorDllList.end())
    {
      if (ulConnectorIdx == ulSearchIdx)
      {
        // Copy UUID, filename, full file name and identifier
        strcpy( ptConnector->szConnectorUUID, (*iterConnector)->szConnectorUUID); 
        strcpy( ptConnector->szFileName, (*iterConnector)->szFileName);
        strcpy( ptConnector->szFullFileName, (*iterConnector)->szFullFileName);
        strcpy( ptConnector->szIdentifier, (*iterConnector)->szConIdentifier);                    
        strcpy( ptConnector->szDescription, (*iterConnector)->szDescription);                    
        ptConnector->ulConnectorType = (*iterConnector)->tType.ulType;
        break;
      }
      ulSearchIdx++;
      iterConnector++;
    }
  }
  
  return lRet;
}

/*! **************************************************************************
* Get connector specific confguration
*   \param  szUUID          String of connector UUID
*   \param  pulConfigSize   size of supplied buffer, returns actual size of config string
*   \param  pcConfig        Returned configuration string
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxConGetConfig (char*          szUUID, 
                                 uint32_t*      pulConfigSize, 
                                 char*          pcConfig)
{
  // Check parameter
  if ((NULL == pulConfigSize) || (NULL == szUUID))
    return NXAPI_INVALID_POINTER;

  // Handle configuration
  int32_t lRet = NXAPI_NO_ERROR;

  // Search for available connector
  NX_CONNECTOR_DLL_LIST::iterator iterConnector = g_cvConnectorDllList.begin();
  while (iterConnector != g_cvConnectorDllList.end())
  {
    // compare UUID string
    if(0 == strcmp(szUUID, (*iterConnector)->szConnectorUUID))
      break;
    
    iterConnector++;
  }

  if(iterConnector == g_cvConnectorDllList.end())
  {
    lRet = NXAPI_NO_ENTRY_FOUND;

  } else if(NULL == (*iterConnector)->hDll)
  {
    lRet = NXAPI_DRIVER_DLL_NOT_LOADED;

  } else if (NULL == pcConfig)
  {
    lRet = (*iterConnector)->tFncTable.pfnNetXConGetConfig( eCMD_CONFIG_GETLEN, (void*)pulConfigSize);

  } else
  {
    uint32_t ulSize = 0;
    lRet = (*iterConnector)->tFncTable.pfnNetXConGetConfig( eCMD_CONFIG_GETLEN, &ulSize);
    if (ulSize>*pulConfigSize)
    {
      lRet = NXAPI_BUFFER_TOO_SHORT;
    } else
    {
      lRet = (*iterConnector)->tFncTable.pfnNetXConGetConfig( eCMD_CONFIG_GETSTRING, pcConfig);
    }
    *pulConfigSize = ulSize;
  }

  return lRet;
}

/*! **************************************************************************
* Set connector specific configuration
*   \param  szUUID       UUID of the connector
*   \param  pcConfig     String of configuration
*   \return NXAPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxConSetConfig ( char* szUUID, char* pcConfig)
{
  if( (NULL == szUUID)  ||
      (NULL == pcConfig) )
    return CIFX_INVALID_POINTER;

  int32_t lRet = NXAPI_NO_ENTRIES;
                   
  // Search for available connector
  NX_CONNECTOR_DLL_LIST::iterator iterConnector = g_cvConnectorDllList.begin();
  while (iterConnector != g_cvConnectorDllList.end())
  {
    // compare UUID string
    if(0 == strcmp(szUUID, (*iterConnector)->szConnectorUUID))
      break;
    
    iterConnector++;
  }

  if(iterConnector != g_cvConnectorDllList.end())
  {
    // Handle state of connector layer and interface 
    if(NULL == (*iterConnector)->hDll)
    {
      lRet = NXAPI_DRIVER_DLL_NOT_LOADED;
    } else
    {
      // Set new configuration
      lRet = (*iterConnector)->tFncTable.pfnNetXConSetConfig( eCMD_CONFIG_SETSTRING, pcConfig);
    }
  }

  return lRet;
}

/*! **************************************************************************
* Get corresponding name information
*   \param  pszSourceName    Name of a device / interface 
*   \param  ulCmd            NXCON_GET_FULL_NAME or NXCON_GET_SHORT_NAME
*   \param  ulCorrespondSize Size of buffer referenced by szCorrespondName
*   \param  szCorrespondName Corresponding name of the device, depends on given command
*   \return NAXPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxConGetCorrespondName ( char* pszSourceName, uint32_t ulCmd, uint32_t ulCorrespondSize, char* szCorrespondName)
{
  if((NULL == pszSourceName)   ||
     (NULL == szCorrespondName) )
     return NXAPI_INVALID_POINTER;

  // Check lists 
  if(g_cvConnectorDllList.empty())
    return NXAPI_NO_ENTRIES;

  if(NULL == g_pcDeviceHandler)
    return NXAPI_DRIVER_NOT_INITIALIZED;
  
  int32_t lRet = NXAPI_NO_ERROR;

  // Search connection use identifier
  bool  fFound = false;
  switch(ulCmd)
  {
    case NXCON_GET_FULL_NAME:  // Get the device 
    {
      std::string szInterface;
      std::string szBoard;

      if(CDeviceHandler::SplitDeviceName(pszSourceName, szInterface, szBoard))
      {
        std::string szLayerPrefix;

        szLayerPrefix = szInterface.substr(0, 3);

        CPhysicalLayer* pcLayer = g_pcDeviceHandler->GetLayer(szLayerPrefix.c_str());

        if( (NULL !=  pcLayer) &&
            (NXAPI_NO_ERROR == (lRet = pcLayer->GetLongInterfaceName(ulCorrespondSize, szCorrespondName, pszSourceName))) )
        {
          fFound = true;
        }
      }
    }
    break;

    case NXCON_GET_SHORT_NAME: // Get short device name from human readable name 
    { 
      CPhysicalLayer* pcLayer;
      uint32_t   ulLayer = 0;

      while(NULL != (pcLayer = g_pcDeviceHandler->GetLayer(ulLayer++)))
      {
        if(NXAPI_NO_ERROR == (lRet = pcLayer->GetShortInterfaceName(ulCorrespondSize, szCorrespondName, pszSourceName)))
        {
          fFound = true;
          break;
        }
      }
    }
    break;

    default:
    {
      lRet = NXAPI_DRIVER_INVALID_COMMAND;
    }
    break;
  }

  // Check if the device connection was not found
  if(!fFound)
  {
    lRet = NXAPI_NO_ENTRY_FOUND;      
  }

  return lRet;
}

/*! **************************************************************************
* Create the configuration dialog of the connector
*   \param  szUUID      UUID of the connector
*   \param  pvParentWnd Parent window
*   \param  pvDialogWnd Window handle of created dialog
*   \return NAXPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxConCreateConfigDialog( char* szUUID, void* pvParentWnd, void** pvDialogWnd)
{
  // Check incomming parameter
  if ( (NULL == pvParentWnd) ||
       (NULL == pvDialogWnd) ||
       (NULL == szUUID)        )
    return NXAPI_INVALID_POINTER;
  
  // Check lists 
  if(g_cvConnectorDllList.empty())
    return NXAPI_NO_ENTRIES;
  
  int32_t lRet = NXAPI_NO_ERROR;

  // Search for available connector
  NX_CONNECTOR_DLL_LIST::iterator iterConnector = g_cvConnectorDllList.begin();
  while (iterConnector != g_cvConnectorDllList.end())
  {
    // compare UUID string
    if(0 == strcmp(szUUID, (*iterConnector)->szConnectorUUID))
      break;
    
    iterConnector++;
  }

  if(iterConnector != g_cvConnectorDllList.end())
  {
    // Handle state of connector layer and interface 
    if(NULL == (*iterConnector)->hDll)
    {
      lRet = NXAPI_DRIVER_DLL_NOT_LOADED;
    } else
    {
      uint32_t ulSize   = 0;

      if (NXCON_NO_ERROR == (*iterConnector)->tFncTable.pfnNetXConGetConfig( eCMD_CONFIG_GETLEN, &ulSize))
      {
        char* szConfig = new char[ulSize+1];
        if (NXCON_NO_ERROR == (*iterConnector)->tFncTable.pfnNetXConGetConfig( eCMD_CONFIG_GETSTRING, szConfig))
        {
          *pvDialogWnd = (*iterConnector)->tFncTable.pfnNetXConCreateDialog( (HWND)pvParentWnd, szConfig);
        }

        delete [] szConfig;
      }

      if(NULL == *pvDialogWnd)
      {
        // CreateDialog failt
        lRet = NXAPI_DRIVER_NOT_INITIALIZED;
      }
    }
  }
  return lRet;
}

/*! **************************************************************************
* Close the configuration dialog of the connector
*   \param  szUUID       UUID of the connector
*   \param  fSaveChanges Store new configuration
*   \return NAXPI_NO_ERROR on success
******************************************************************************/
int32_t APIENTRY nxConCloseConfigDialog( char* szUUID, BOOL fSaveChanges)
{
  // Check incomming parameter
  if (NULL == szUUID)
    return NXAPI_INVALID_POINTER;
  
  // Check lists 
  if(g_cvConnectorDllList.empty())
    return NXAPI_NO_ENTRIES;
  
  int32_t lRet = NXAPI_NO_ERROR;

  // Search for available connector
  NX_CONNECTOR_DLL_LIST::iterator iterConnector = g_cvConnectorDllList.begin();
  while (iterConnector != g_cvConnectorDllList.end())
  {
    // compare UUID string
    if(0 == strcmp(szUUID, (*iterConnector)->szConnectorUUID))
      break;
    
    iterConnector++;
  }

  if(iterConnector != g_cvConnectorDllList.end())
  {
    // Handle state of connector layer and interface 
    if(NULL == (*iterConnector)->hDll)
    {
      lRet = NXAPI_DRIVER_DLL_NOT_LOADED;
    } else
    {
      if (fSaveChanges)
      {
        uint32_t ulSize = 0;
        if (NXCON_NO_ERROR == (*iterConnector)->tFncTable.pfnNetXConGetConfig( eCMD_DIALOG_GETLEN, &ulSize))
        {
          char* szConfig = new char[ulSize+1];
          if (NXCON_NO_ERROR == (*iterConnector)->tFncTable.pfnNetXConGetConfig( eCMD_DIALOG_GETSTRING, szConfig))
          {
            (*iterConnector)->tFncTable.pfnNetXConSetConfig(eCMD_CONFIG_SETSTRING, szConfig);
          }
          delete [] szConfig;
        }
      }

      (*iterConnector)->tFncTable.pfnNetXConEndDialog();
    }
  }

  return lRet;
}
